import os
import requests
from urllib.parse import urlparse
from pathlib import Path

# === CSV failu saraksts ===
# Šeit ir saraksts ar CSV failiem, kas nepieciešami "Pensionārs" moduļa funkcionalitātei
csv_urls = [
    # Uzņēmumu reģistra pamatdati
    "https://data.gov.lv/dati/dataset/4de9697f-850b-45ec-8bba-61fa09ce932f/resource/25e80bf3-f107-4ab4-89ef-251b5b9374e9/download/register.csv",
    # Patiesā labuma guvēji
    "https://data.gov.lv/dati/dataset/b7848ab9-7886-4df0-8bc6-70052a8d9e1a/resource/20a9b26d-d056-4dbb-ae18-9ff23c87bdee/download/beneficial_owners.csv",
    # Amatpersonas
    "https://data.gov.lv/dati/dataset/096c7a47-33cd-4dc9-a876-2c86e86230fd/resource/e665114a-73c2-4375-9470-55874b4cfa6b/download/officers.csv",
    # VID dati par nodokļiem, darbiniekiem un NACE kodiem
    "https://data.gov.lv/dati/dataset/5ed74664-b49d-4b28-aacb-040931646e9b/resource/a42d6e8c-1768-4939-ba9b-7700d4f1dd3a/download/pdb_nm_komersantu_samaksato_nodoklu_kopsumas_odata.csv",
    # Finanšu pārskati (galvenā informācija)
    "https://data.gov.lv/dati/dataset/8d31b878-536a-44aa-a013-8bc6b669d477/resource/27fcc5ec-c63b-4bfd-bb08-01f073a52d04/download/financial_statements.csv",
    # Peļņas vai zaudējumu aprēķini
    "https://data.gov.lv/dati/dataset/8d31b878-536a-44aa-a013-8bc6b669d477/resource/d5fd17ef-d32e-40cb-8399-82b780095af0/download/income_statements.csv",
    # Bilances
    "https://data.gov.lv/dati/dataset/8d31b878-536a-44aa-a013-8bc6b669d477/resource/50ef4f26-f410-4007-b296-22043ca3dc43/download/balance_sheets.csv",
    # Akcionāri
    "https://data.gov.lv/dati/dataset/8a66567b-3583-4aa0-841b-eb13f025cd78/resource/6adabd83-93f9-4d7f-bebd-fa109bbf794a/download/stockholders.csv",
    "https://data.gov.lv/dati/dataset/8a66567b-3583-4aa0-841b-eb13f025cd78/resource/55370dca-6765-4566-b2cc-42537782f750/download/stockholders_joint_owners.csv",
    # Dalībnieki
    "https://data.gov.lv/dati/dataset/e1162626-e02a-4545-9236-37553609a988/resource/837b451a-4833-4fd1-bfdd-b45b35a994fd/download/members.csv",
    "https://data.gov.lv/dati/dataset/e1162626-e02a-4545-9236-37553609a988/resource/466e5b9b-b31b-4e68-aaee-ae34254b902b/download/members_joint_owners.csv",
    # Nodokļu maksātāju reitings
    "https://data.gov.lv/dati/dataset/41481e3e-630f-4b73-b02e-a415d27896db/resource/acd4c6f9-5123-46a5-80f6-1f44b4517f58/download/reitings_uznemumi.csv",
]

# === Galvenā funkcija ===
def download_csv_files(urls):
    """
    Lejupielādē CSV failus no norādītajiem URL.
    Faili tiek saglabāti 'csv' apakšdirektorijā relatīvi pret skripta atrašanās vietu.
    """
    # Nosakām ceļu uz 'csv' mapi, kas atrodas tajā pašā direktorijā, kur skripts
    script_dir = Path(__file__).parent
    dir_name = script_dir / "csv"

    # Izveido direktoriju (ja tāda vēl neeksistē)
    try:
        os.makedirs(dir_name, exist_ok=True)
        print(f"Direktorija '{dir_name}' ir gatava.")
    except OSError as e:
        print(f"Kļūda veidojot direktoriju '{dir_name}': {e}")
        return

    # Izdzēš visus esošos .csv failus direktorijā
    print(f"Dzēš visus esošos .csv failus no direktorijas '{dir_name}'...")
    for file in os.listdir(dir_name):
        if file.endswith(".csv"):
            file_path = dir_name / file
            try:
                os.remove(file_path)
                print(f"  Izdzēsts: {file}")
            except OSError as e:
                print(f"Kļūda dzēšot failu {file}: {e}")
                
    # Iterē cauri katram URL un lejupielādē failu
    download_count = 0
    error_count = 0
    
    unique_urls = sorted(list(set(urls)))

    for url in unique_urls:
        try:
            # Iegūst faila nosaukumu no URL
            parsed_url = urlparse(url)
            filename = os.path.basename(parsed_url.path)
            
            file_path = dir_name / filename

            print(f"Lejupielādē: {filename}...")

            # Veic HTTP GET pieprasījumu
            response = requests.get(url, stream=True, timeout=60)
            response.raise_for_status()

            # Saglabā faila saturu
            with open(file_path, 'wb') as f:
                for chunk in response.iter_content(chunk_size=8192):
                    f.write(chunk)

            print(f"Veiksmīgi saglabāts: {file_path}")
            download_count += 1

        except requests.exceptions.RequestException as e:
            print(f"Kļūda lejupielādējot {url}: {e}")
            error_count += 1
        except Exception as e:
            print(f"Nezināma kļūda apstrādājot {url}: {e}")
            error_count += 1

    # Paziņo par rezultātu
    print("\n--- Lejupielādes kopsavilkums ---")
    print(f"Lejupielādēti {download_count} jauni faili.")
    print(f"Faili ar kļūdām: {error_count}.")
    print("---------------------------------")

# === Skripta izpilde ===
if __name__ == "__main__":
    download_csv_files(csv_urls)
    print("Datu lejupielāde pabeigta.")
