# projects/python_generator/processing/loader.py

import pandas as pd
from pathlib import Path

def load_all_csv_data(csv_dir: Path, table_names_list: list):
    """
    Ielādē visus norādītos CSV failus no norādītās direktorijas kā pandas DataFrames.
    Mēģina automātiski noteikt atdalītāju un veic pamata datu tīrīšanu.
    """
    dataframes = {}
    print(f"Mēģina ielādēt CSV failus no: {csv_dir}")

    # Zināmie atdalītāji katrai tabulai, lai paātrinātu procesu
    known_delimiters = {
        'register': ';',
        'financial_statements': ';',
        'income_statements': ';',
        'balance_sheets': ';',
        'cash_flow_statements': ';',
        'areas_of_activity_of_associations_foundations': ';',
        'liquidations': ';',
        'insolvency_legal_person_proceeding': ';',
        'arbitration_list': ';',
        'arbitration_members': ';',
        'area_of_activity': ';',
        'political_parties': ';',
        'property_investment_appraisers_list': ';',
        'religious_affiliations': ';',
        'special_statuses': ';',
        'register_name_history': ';',
        'stockholders_joint_owners': ',',
        'beneficial_owners': ';',
        'members': ';',
        'equity_capitals': ';',
        'stockholders': ';',
        'reorganizations': ';',
        'suspensions_prohibitions': ';',
        'ppi_public_persons_institutions': ';',
        'memorandum_date': ';',
        'officers': ';',
        'ppi_delegated_entities': ';',
        'securing_measures': ';',
        'akf_data': ';',
        'members_joint_owners': ';',
        'farm_land': ';',
        # === SĀKUMS: JAUNO FAILU PIEVIENOŠANA ===
        'pdb_nm_komersantu_samaksato_nodoklu_kopsumas_odata': ',',
        'pdb_pvnmaksataji_odata': ',',
        'reitings_uznemumi': ','
        # === BEIGAS: JAUNO FAILU PIEVIENOŠANA ===
    }

    # Kolonnas, kuras vienmēr jāielādē kā teksts (string), lai izvairītos no
    # automātiskas konvertēšanas uz skaitļiem (piem., `00123` -> `123`).
    always_string_cols = [
        'id', 'statement_id', 'file_id', 'regcode',
        'legal_entity_registration_number',
        'at_legal_entity_registration_number', 'member_id', 'super_id',
        # === SĀKUMS: JAUNO FAILU PIEVIENOŠANA ===
        'Registracijas_kods', 'Numurs'
        # === BEIGAS: JAUNO FAILU PIEVIENOŠANA ===
    ]

    for table_name in table_names_list:
        file_path = csv_dir / f"{table_name}.csv"
        print(f"\nApstrādā failu: {file_path}")
        if not file_path.exists():
            print(f"Brīdinājums: Fails netika atrasts: {file_path}")
            dataframes[table_name] = pd.DataFrame()
            continue

        df = None
        # Mēģināmie atdalītāji, sākot ar zināmo, ja tāds ir
        delimiters_to_try = []
        if table_name in known_delimiters:
            delimiters_to_try.append(known_delimiters[table_name])
        for d_default in [';', ',', '\t', '|']:
            if d_default not in delimiters_to_try:
                delimiters_to_try.append(d_default)

        for delim in delimiters_to_try:
            try:
                print(f"  Mēģina atdalītāju '{delim}'...")
                # Mēģinām ar ātrāko 'c' dzinēju
                df_attempt = pd.read_csv(
                    file_path,
                    sep=delim,
                    low_memory=False,
                    encoding='utf-8',
                    on_bad_lines='warn',
                    engine='c',
                    dtype={col: str for col in always_string_cols if col is not None}
                )
                # Ja rezultātā ir tikai viena kolonna, tas, visticamāk, ir nepareizs atdalītājs
                if len(df_attempt.columns) > 1:
                    df = df_attempt
                    print(f"    -> Veiksmīgi nolasīts ar atdalītāju '{delim}'. Kolonnas: {len(df.columns)}")
                    break
            except Exception as e_c:
                print(f"    -> Kļūda ar 'c' dzinēju: {e_c}. Mēģina 'python' dzinēju.")
                try:
                    # Ja 'c' dzinējs neizdodas, mēģinām lēnāko, bet elastīgāko 'python' dzinēju
                    df_attempt = pd.read_csv(
                        file_path,
                        sep=delim,
                        low_memory=False,
                        engine='python',
                        quotechar='"',
                        escapechar='\\',
                        dtype={col: str for col in always_string_cols if col is not None}
                    )
                    if len(df_attempt.columns) > 1:
                        df = df_attempt
                        print(f"    -> Veiksmīgi nolasīts ar atdalītāju '{delim}' (dzinējs 'python'). Kolonnas: {len(df.columns)}")
                        break
                except Exception as e_py:
                    print(f"    -> Kļūda ar 'python' dzinēju: {e_py}")
                    continue # Mēģinām nākamo atdalītāju

        if df is None or df.empty:
            print(f"  -> KĻŪDA: Nevarēja efektīvi nolasīt {file_path}. DataFrame ir tukšs.")
            dataframes[table_name] = pd.DataFrame()
            continue

        # Papildu datu tīrīšana un tipu konvertēšana
        df.dropna(axis=1, how='all', inplace=True) # Dzēšam kolonnas, kurās ir tikai tukšas vērtības

        # Konvertējam identifikatorus uz string, ja tie nav jau konvertēti
        for id_c in always_string_cols:
            if id_c in df.columns:
                df[id_c] = df[id_c].fillna('').astype(str)
        
        # === SĀKUMS: SPECIĀLĀ APSTRĀDE PVN MAKSĀTĀJU FAILAM ===
        if table_name == 'pdb_pvnmaksataji_odata' and 'Numurs' in df.columns:
            print("  -> Veic speciālo apstrādi 'pdb_pvnmaksataji_odata.csv': noņem 'LV' prefiksu.")
            df['Numurs'] = df['Numurs'].str.replace('LV', '', regex=False).str.strip()
        # === BEIGAS: SPECIĀLĀ APSTRĀDE PVN MAKSĀTĀJU FAILAM ===

        dataframes[table_name] = df
        cols_to_show = list(df.columns)[:5]
        if len(df.columns) > 5:
            cols_to_show.append("...")
        print(f"  -> Pabeigta ielāde: {table_name}. Rindas: {len(df)}, Kolonnas: {len(df.columns)} ({cols_to_show})")

    return dataframes