# projects/python_generator/core/page_builder.py

import pandas as pd
import json
from processing.formatters import get_raw_value, sanitize_for_json, format_number_data
from config.translations import COLUMN_NAME_TRANSLATIONS
from config.table_definitions import TABLE_DISPLAY_CONFIG
from config.settings import (
    MIN_AUTOCOMPLETE_CHARS, MAX_SEARCH_TERM_LENGTH, ASSETS_BASE_PATH_FOR_HTML,
    BASE_DOMAIN
)

# Konfigurācija datu maskēšanai
MASKING_CONFIG = {
    'members': ['latvian_identity_number_masked', 'birth_date'],
    'officers': ['latvian_identity_number_masked', 'birth_date'],
    'beneficial_owners': ['latvian_identity_number_masked', 'birth_date'],
    'stockholders': ['latvian_identity_number_masked', 'birth_date'],
    'members_joint_owners': ['latvian_identity_number_masked', 'birth_date'],
    'stockholders_joint_owners': ['latvian_identity_number_masked', 'birth_date'],
}

def _calculate_net_salary(gross_salary, year):
    """
    Helper funkcija, kas aprēķina neto algu, ņemot vērā Latvijas nodokļu nianses.
    """
    if gross_salary is None or gross_salary <= 0:
        return {'net_salary': None, 'non_taxable_minimum': 0, 'iin': 0, 'vsaoi_employee': 0}

    params = {
        2022: {'vsaoi_rate': 0.105, 'iin_rates': [(20004, 0.20), (78100, 0.23), (float('inf'), 0.31)], 'non_taxable_formula': lambda gs: max(0, 350 - 0.5 * (gs - 500)) if 500 < gs <= 1200 else (350 if gs <= 500 else 0)},
        2023: {'vsaoi_rate': 0.105, 'iin_rates': [(20004, 0.20), (78100, 0.23), (float('inf'), 0.31)], 'non_taxable_formula': lambda gs: max(0, 500 - 0.38462 * (gs - 500)) if 500 < gs <= 1800 else (500 if gs <= 500 else 0)},
        2024: {'vsaoi_rate': 0.105, 'iin_rates': [(20004, 0.20), (78100, 0.23), (float('inf'), 0.31)], 'non_taxable_formula': lambda gs: max(0, 500 - 0.38462 * (gs - 500)) if 500 < gs <= 1800 else (500 if gs <= 500 else 0)},
        2025: {'vsaoi_rate': 0.105, 'iin_rates': [(20004, 0.20), (78100, 0.23), (float('inf'), 0.31)], 'non_taxable_formula': lambda gs: max(0, 500 - 0.38462 * (gs - 500)) if 500 < gs <= 1800 else (500 if gs <= 500 else 0)}
    }
    
    year_params = params.get(year, params[2024])

    vsaoi_employee = gross_salary * year_params['vsaoi_rate']
    non_taxable_minimum = year_params['non_taxable_formula'](gross_salary)
    
    taxable_income = gross_salary - vsaoi_employee - non_taxable_minimum
    if taxable_income < 0:
        taxable_income = 0
    
    annual_gross = gross_salary * 12
    iin = 0
    
    if annual_gross <= year_params['iin_rates'][0][0]:
        iin = taxable_income * year_params['iin_rates'][0][1]
    elif annual_gross <= year_params['iin_rates'][1][0]:
        iin = taxable_income * year_params['iin_rates'][1][1]
    else:
        iin = taxable_income * year_params['iin_rates'][2][1]
         
    net_salary = gross_salary - vsaoi_employee - iin

    return {
        'net_salary': round(net_salary, 2),
        'non_taxable_minimum': round(non_taxable_minimum, 2),
        'iin': round(iin, 2),
        'vsaoi_employee': round(vsaoi_employee, 2)
    }

def prepare_vid_panel_data(results: dict):
    panel_data = {
        'has_data': False,
        'rating': None,
        'pvn': None,
        'tax_table': [],
        'salary_calculation_example': None
    }

    def _parse_vid_number(value_str):
        if not value_str or not isinstance(value_str, str):
            return 0.0
        try:
            cleaned_str = value_str.replace(' ', '').replace('\xa0', '').replace(',', '.')
            return float(cleaned_str)
        except (ValueError, TypeError):
            return 0.0

    rating_data = results.get('reitings_uznemumi', [])
    if rating_data:
        panel_data['rating'] = rating_data[0]
        panel_data['has_data'] = True

    pvn_data = results.get('pdb_pvnmaksataji_odata', [])
    if pvn_data:
        panel_data['pvn'] = pvn_data[0]
        panel_data['has_data'] = True

    tax_data_raw = results.get('pdb_nm_komersantu_samaksato_nodoklu_kopsumas_odata', [])
    if tax_data_raw:
        panel_data['has_data'] = True
        
        tax_data = sorted(tax_data_raw, key=lambda x: int(x.get('Taksacijas_gads', 0)), reverse=True)
        
        for i, row in enumerate(tax_data):
            try:
                year_val = row.get('Taksacijas_gads')
                vsaoi_val = row.get('Taja_skaita_VSAOI')
                employees_val = row.get('Videjais_nodarbinato_personu_skaits_cilv')

                if not all([year_val, vsaoi_val, employees_val]):
                    continue

                year = int(year_val)
                vsaoi_total_thousands = _parse_vid_number(vsaoi_val)
                employees = int(float(employees_val))
                
                avg_gross_salary = 0
                salary_details = {}

                if employees > 0 and vsaoi_total_thousands > 0:
                    avg_gross_salary = ((vsaoi_total_thousands * 1000) / 0.3409) / employees / 12
                    salary_details = _calculate_net_salary(avg_gross_salary, year)

                new_row = {
                    'Taksacijas_gads': year,
                    'Samaksato_VID_administreto_nodoklu_kopsumma_tukst_EUR': row.get('Samaksato_VID_administreto_nodoklu_kopsumma_tukst_EUR'),
                    'Taja_skaita_IIN': row.get('Taja_skaita_IIN'),
                    'Taja_skaita_VSAOI': row.get('Taja_skaita_VSAOI'),
                    'Videjais_nodarbinato_personu_skaits_cilv': employees,
                    'avg_gross_salary': f"{avg_gross_salary:,.2f}".replace(",", " ") if avg_gross_salary > 0 else "—",
                    'avg_net_salary': f"~ {salary_details.get('net_salary'):,.2f}".replace(",", " ") if salary_details.get('net_salary') else "—"
                }
                panel_data['tax_table'].append(new_row)

                if i == 0 and avg_gross_salary > 0:
                    panel_data['salary_calculation_example'] = {
                        'year': year,
                        'vsaoi_sum': vsaoi_total_thousands,
                        'employees': employees,
                        'gross_salary_raw': avg_gross_salary,
                        'gross_salary_formatted': new_row['avg_gross_salary'],
                        'net_salary_formatted': new_row['avg_net_salary'],
                        'vsaoi_employee_part': f"{salary_details.get('vsaoi_employee'):,.2f}".replace(",", " "),
                        'non_taxable_minimum': f"{salary_details.get('non_taxable_minimum'):,.2f}".replace(",", " "),
                        'iin_part': f"{salary_details.get('iin'):,.2f}".replace(",", " ")
                    }

            except (ValueError, TypeError, ZeroDivisionError) as e:
                continue
                
    return panel_data

def prepare_faq_data(page_data: dict):
    segment = page_data.get('segment', {})
    company_title = page_data.get('companyTitleForHtml', 'Šis uzņēmums')
    vid_data = page_data.get('vid_panel_data', {})
    nace_description = page_data.get('nace_description')
    faq_list = []

    status = segment.get('status')
    form_group = segment.get('form_group')
    
    address = page_data.get('formattedAddressForHtml')
    if address and address != '—':
        address_question = f"Kāda ir {company_title} juridiskā adrese?"
        if status == 'Likvidēts':
            address_question = f"Kāda bija {company_title} pēdējā reģistrētā juridiskā adrese?"
        faq_list.append({"question": address_question, "answer": f"Juridiskā adrese ir {address}."})

    if status == 'Likvidēts':
        date = page_data.get('liquidation_date', 'nezināms')
        faq_list.insert(0, { 
            "question": f"Vai uzņēmums {company_title} ir aktīvs?",
            "answer": f"Nē, šī uzņēmuma darbība ir izbeigta {date}."
        })
    
    elif status == 'Aktīvs':
        if form_group == 'Komercsabiedrība':
            summary_data_ugp = page_data.get('summary_table_data_for_js', {}).get('UGP', [])
            latest_report = summary_data_ugp[0] if summary_data_ugp else {}
            if latest_report:
                year = latest_report.get('year')
                profit = latest_report.get('profit')
                turnover = latest_report.get('turnover')
                employees = latest_report.get('employees')

                if all(v is not None for v in [year, profit, turnover]):
                    turnover_str = format_number_data(turnover, latest_report.get('currency', 'EUR'))
                    
                    if profit >= 0:
                        profit_loss_text = "peļņa bija"
                        profit_str = format_number_data(profit, latest_report.get('currency', 'EUR'))
                    else:
                        profit_loss_text = "zaudējumi bija"
                        profit_str = format_number_data(abs(profit), latest_report.get('currency', 'EUR'))
                    
                    faq_list.append({
                        "question": f"Kāds bija {company_title} {year}. gada apgrozījums un finanšu rezultāts?",
                        "answer": f"{year}. gadā uzņēmuma apgrozījums bija {turnover_str}, un {profit_loss_text} {profit_str}."
                    })
                
                if all(v is not None for v in [year, employees]):
                    faq_list.append({
                        "question": f"Cik darbinieku bija uzņēmumam {company_title} {year}. gadā?",
                        "answer": f"{year}. gadā uzņēmumā vidēji bija nodarbināti {int(employees)} darbinieki."
                    })

    name_history = page_data.get('results', {}).get('register_name_history', [])
    if name_history:
        history_names = ", ".join([f"\"{h.get('name', '')}\"" for h in name_history])
        faq_list.append({
            "question": f"Kādi ir bijuši iepriekšējie {company_title} nosaukumi?",
            "answer": f"Uzņēmuma iepriekšējie nosaukumi ir bijuši: {history_names}."
        })
        
    pvn_info = vid_data.get('pvn')
    if pvn_info and pvn_info.get('Aktivs'):
        pvn_statuss = "ir" if pvn_info.get('Aktivs') == 'ir' else "nav"
        atbilde = f"Saskaņā ar VID publiskajiem datiem, uzņēmums {pvn_statuss} reģistrēts kā aktīvs PVN maksātājs."
        faq_list.append({
            "question": f"Vai {company_title} ir PVN maksātājs?",
            "answer": atbilde
        })

    tax_table = vid_data.get('tax_table')
    if tax_table:
        latest_salary_data = tax_table[0]
        year = latest_salary_data.get('Taksacijas_gads')
        gross_salary = latest_salary_data.get('avg_gross_salary')
        net_salary = latest_salary_data.get('avg_net_salary')
        if year and gross_salary and gross_salary != "—":
            atbilde = (f"Balstoties uz VID datiem par {year}. gadu, aprēķinātā vidējā bruto alga (pirms nodokļiem) "
                       f"uzņēmumā bija aptuveni {gross_salary} EUR mēnesī. Tas atbilst aptuvenai neto algai "
                       f"('uz rokas') ap {net_salary} EUR. Jāņem vērā, ka šis ir aptuvens aprēķins.")
            faq_list.append({
                "question": f"Kāda ir vidējā alga uzņēmumā {company_title}?",
                "answer": atbilde
            })
            
    if nace_description:
        atbilde = f"Uzņēmuma galvenā reģistrētā darbības nozare saskaņā ar NACE klasifikatoru ir: {nace_description}."
        faq_list.append({
            "question": f"Kāda ir {company_title} galvenā darbības nozare?",
            "answer": atbilde
        })
        
    return faq_list

def get_company_details_for_panel(company_main_data: pd.Series, search_reg_nr: str, segment: dict):
    details = {
        'companyTitleForHtml': str(search_reg_nr),
        'formattedAddressForHtml': '—',
        'statusText': 'Nezināms',
        'statusClass': 'status-unknown-bg',
        'terminatedDisplay': '—',
        'closedDisplay': '—',
        'closedClassModifier': 'value-no-data-black',
        'is_liquidated': False,
        'liquidation_date': None,
        'show_financial_charts': True,
        'show_summary_panel': True
    }

    if company_main_data is None or company_main_data.empty:
        details['statusText'] = 'Dati nav atrasti'
        details['show_financial_charts'] = False
        details['show_summary_panel'] = False
        return details

    if segment.get('status') in ['Likvidēts', 'Reorganizēts', 'Cits']:
        details['is_liquidated'] = True
        details['liquidation_date'] = get_raw_value(company_main_data, 'terminated') or get_raw_value(company_main_data, 'closed')
        details['show_financial_charts'] = False
        details['show_summary_panel'] = False

    if segment.get('form_group') in ['NVO', 'Partija']:
        details['show_financial_charts'] = False
        details['show_summary_panel'] = False

    type_code = get_raw_value(company_main_data, 'type')
    name_in_quotes = get_raw_value(company_main_data, 'name_in_quotes')
    full_name = get_raw_value(company_main_data, 'name')
    prefix_type_codes = ['SIA', 'AS', 'IK', 'ZS', 'ZEM', 'PS', 'PA', 'AROD']
    company_title = str(search_reg_nr)

    if name_in_quotes:
        prefix = None
        if type_code and str(type_code) in prefix_type_codes:
            prefix = 'ZS' if str(type_code) == 'ZEM' else str(type_code)
        company_title = f'{prefix} "{name_in_quotes}"' if prefix else f'"{name_in_quotes}"'
    elif full_name:
        company_title = str(full_name).replace('""', '"')
    details['companyTitleForHtml'] = company_title
    
    address = get_raw_value(company_main_data, 'address')
    index_val = get_raw_value(company_main_data, 'index')
    if address:
        parts = [p.strip() for p in str(address).replace('""', '"').split(',') if p.strip()]
        if parts:
            address_parts = list(reversed(parts))
            if index_val and str(index_val).strip() and str(index_val) != '0':
                address_parts.append(f'LV-{str(index_val).split(".")[0]}')
            details['formattedAddressForHtml'] = ', '.join(address_parts)

    closed_val_raw = get_raw_value(company_main_data, 'closed')
    terminated_date_raw = get_raw_value(company_main_data, 'terminated')
    
    is_active_status_flag = True

    if closed_val_raw == 'L':
        details['statusText'] = "Likvidēts"
        details['statusClass'] = "status-inactive-bg"
        is_active_status_flag = False
        details['closedDisplay'] = "Likvidēts"
        details['closedClassModifier'] = "closed-l"
    elif closed_val_raw == 'R':
        details['statusText'] = "Reorganizēts"
        details['statusClass'] = "status-process-bg"
        is_active_status_flag = False
        details['closedDisplay'] = "Reorganizēts"
        details['closedClassModifier'] = "closed-r"
    elif terminated_date_raw and str(terminated_date_raw).strip() and str(terminated_date_raw) != '0000-00-00':
        details['statusText'] = "Darbība izbeigta"
        details['statusClass'] = "status-process-bg"
        is_active_status_flag = False
        details['closedClassModifier'] = 'value-no-data-black'
    else:
        details['statusText'] = "Aktīvs"
        details['statusClass'] = "status-active-bg"
        is_active_status_flag = True
        details['closedDisplay'] = '—'
        details['closedClassModifier'] = 'value-no-data-black'

    if terminated_date_raw and str(terminated_date_raw).strip() and str(terminated_date_raw) != '0000-00-00':
        details['terminatedDisplay'] = str(terminated_date_raw)
    elif is_active_status_flag:
        details['terminatedDisplay'] = '—'
    else:
        details['terminatedDisplay'] = '—'

    if is_active_status_flag:
        details['closedDisplay'] = '—'
        details['closedClassModifier'] = 'value-no-data-black'

    return details

def prepare_seo_metadata(page_data: dict):
    segment = page_data.get('segment', {})
    company_main = page_data.get('dati_php_rowData')
    search_reg_nr = page_data.get('search_reg_nr')
    company_title = page_data.get('companyTitleForHtml')
    summary_data_ugp = page_data.get('summary_table_data_for_js', {}).get('UGP', [])
    latest_report = summary_data_ugp[0] if summary_data_ugp else {}

    meta_description = f"Viss par {company_title} (reģ. nr. {search_reg_nr}): statuss, adrese, amatpersonas, finanšu dati."
    status = segment.get('status')
    form_group = segment.get('form_group')
    financials = segment.get('financials')

    if status == 'Likvidēts':
        date = page_data.get('liquidation_date', 'N/A')
        meta_description = f"{company_title} (reģ. nr. {search_reg_nr}) ir likvidēts {date}. Apskatiet vēsturiskos uzņēmuma datus."
    elif status == 'Aktīvs':
        if form_group == 'Komercsabiedrība' and latest_report:
            year = latest_report.get('year')
            profit = latest_report.get('profit')
            turnover = latest_report.get('turnover')
            if financials == 'Peļņa' and profit is not None and turnover is not None:
                meta_description = f"{company_title} ({search_reg_nr}) jaunākie dati. {year}. gada peļņa: {profit:,.0f} EUR pie {turnover:,.0f} EUR apgrozījuma.".replace(",", " ")
            elif financials == 'Zaudējumi' and turnover is not None:
                meta_description = f"{company_title} ({search_reg_nr}) jaunākie dati. {year}. gada apgrozījums: {turnover:,.0f} EUR. Apskatīt pilnu finanšu pārskatu.".replace(",", " ")
        else:
            meta_description = f"{company_title} ({search_reg_nr}) - aktīvs. Visa UR informācija: adrese, statuss, vēsture."
            
    page_data['meta_description'] = meta_description
    page_data['page_keywords'] = ", ".join(filter(None, [company_title, get_raw_value(company_main, 'type_text'), search_reg_nr, 'Uzņēmumu reģistrs', 'finanšu dati']))

    faq_list = prepare_faq_data(page_data)
    page_data['faq_list'] = faq_list

    og_title = page_data.get('current_page_title', 'Saraksts.lv')
    seo_data = {
        'open_graph': {
            'title': og_title,
            'description': page_data.get('meta_description'),
            'type': 'website',
            'url': f"{BASE_DOMAIN}/{search_reg_nr}",
            'image': f"{BASE_DOMAIN}/assets/img/social_logo.png"
        },
        'schema_org_json': None,
        'faq_schema_json': None
    }

    if company_main is not None:
        schema_type = "Organization"
        if form_group == 'NVO': schema_type = "NGO"
        if form_group == 'Partija': schema_type = "PoliticalParty"
        
        address_val = company_main.get('address')
        address_str = str(address_val) if pd.notna(address_val) else ''

        schema = {
            "@context": "https://schema.org",
            "@type": schema_type,
            "name": company_title,
            "legalName": get_raw_value(company_main, 'name'),
            "url": seo_data['open_graph']['url'],
            "logo": seo_data['open_graph']['image'],
            "registrationDate": get_raw_value(company_main, 'registered'),
            "address": {
                "@type": "PostalAddress",
                "streetAddress": address_str.split(',')[1].strip() if ',' in address_str else address_str,
                "addressLocality": address_str.split(',')[0].strip() if ',' in address_str else None,
                "postalCode": get_raw_value(company_main, 'index'),
                "addressCountry": "LV"
            },
            "vatID": f"LV{search_reg_nr}",
            "identifier": {
                "@type": "PropertyValue",
                "propertyID": "Latvian Company Registration Number",
                "value": search_reg_nr
            }
        }
      
        if status == 'Likvidēts' and page_data.get('liquidation_date'):
            schema['dissolutionDate'] = page_data['liquidation_date']
        
        clean_schema = {k: v for k, v in schema.items() if v is not None}
        seo_data['schema_org_json'] = json.dumps(sanitize_for_json(clean_schema), ensure_ascii=False, indent=4)
    
    if faq_list:
        faq_schema_entries = [{"@type": "Question", "name": item["question"], "acceptedAnswer": {"@type": "Answer", "text": item["answer"]}} for item in faq_list]
        faq_page_schema = {"@context": "https://schema.org", "@type": "FAQPage", "mainEntity": faq_schema_entries}
        seo_data['faq_schema_json'] = json.dumps(faq_page_schema, ensure_ascii=False, indent=4)

    return seo_data

def prepare_data_for_results_tables(page_data: dict):
    results = page_data.get('results', {})
    if not results:
        return []
    
    TABLES_TO_EXCLUDE = {
        'officers',
        'beneficial_owners',
        'stockholders',
        'members_joint_owners',
        'stockholders_joint_owners',
        'members'
    }

    current_page_reg_nr = page_data.get('search_reg_nr')
    base_url = f"{BASE_DOMAIN}/"
    linkable_reg_nr_columns = {
        'regcode', 'at_legal_entity_registration_number', 'depository_registration_number',
        'legal_entity_registration_number', 'debtor_registration_number',
        'delegatedEntityRegistrationNumber', 'registrationNumber'
    }

    END_TABLES_CONFIG = {
        'members': 'Dalībnieki (Meklētajā uzņēmumā)',
        'officers': None,
        'beneficial_owners': None,
        'stockholders': 'Akcionāri (Meklētajā uzņēmumā)'
    }
    end_table_keys_ordered = list(END_TABLES_CONFIG.keys())

    main_tables_to_process = []
    end_tables_to_process = []

    main_keys = [k for k in results if k not in end_table_keys_ordered]
    sorted_main_keys = sorted(main_keys, key=lambda k: TABLE_DISPLAY_CONFIG.get(k, {}).get('rank', 999))

    for key in sorted_main_keys:
        if results.get(key) and key not in TABLES_TO_EXCLUDE:
            main_tables_to_process.append({'data': results[key], 'key': key})

    if page_data.get('member_as_entity_data'):
        main_tables_to_process.append({
            'data': page_data['member_as_entity_data'],
            'key': 'members',
            'title': "Dalība citos uzņēmumos (kā subjekts)"
        })
        main_tables_to_process = sorted(main_tables_to_process, key=lambda t: TABLE_DISPLAY_CONFIG.get(t['key'], {}).get('rank', 999))

    for key in end_table_keys_ordered:
        if results.get(key) and key not in TABLES_TO_EXCLUDE:
            table_info = {'data': results[key], 'key': key}
            if END_TABLES_CONFIG[key]:
                table_info['title'] = END_TABLES_CONFIG[key]
            end_tables_to_process.append(table_info)

    tables_to_render = main_tables_to_process + end_tables_to_process
    
    processed_tables = []
    for table_info in tables_to_render:
        data_rows = table_info.get('data')
        if not data_rows or not isinstance(data_rows, list) or not data_rows[0] or not isinstance(data_rows[0], dict):
            continue

        config_key = table_info['key']
        config_entry = TABLE_DISPLAY_CONFIG.get(config_key, {})
        title = table_info.get('title') or config_entry.get('title', config_key.replace('_', ' ').title())

        headers = []
        original_headers = [h for h in data_rows[0].keys() if str(h).lower() not in ['super_id']]
        
        for header in original_headers:
            translation = COLUMN_NAME_TRANSLATIONS.get(str(header), {})
            headers.append({
                'original': str(header),
                'short': translation.get('short', str(header)),
                'full': translation.get('full', '')
            })

        rows = []
        for row_dict in data_rows:
            formatted_cells = []
            for header_info in headers:
                col_name = header_info['original']
                cell_value = row_dict.get(col_name)
                final_value = None

                table_key = table_info['key']
                if table_key in MASKING_CONFIG and col_name in MASKING_CONFIG[table_key]:
                    if pd.notna(cell_value) and str(cell_value).strip() != '':
                        final_value = '@' * len(str(cell_value))
                    else:
                        final_value = cell_value
                else:
                    if not pd.isna(cell_value):
                        if isinstance(cell_value, float) and cell_value == int(cell_value):
                            final_value = int(cell_value)
                        else:
                            final_value = cell_value

                cell_data = {'value': final_value, 'is_link': False, 'url': None}
                value_str = str(final_value).strip()

                if col_name in linkable_reg_nr_columns and value_str.isdigit() and len(value_str) == 11 and value_str != current_page_reg_nr:
                    cell_data['is_link'] = True
                    cell_data['url'] = f"{base_url}{value_str}"
                
                formatted_cells.append(cell_data)

            row_id_val = get_raw_value(row_dict, 'id') or get_raw_value(row_dict, 'statement_id')
            rows.append({
                'values': formatted_cells,
                'data_attributes': {'data-row-id': str(row_id_val)} if row_id_val else {}
            })
        
        processed_tables.append({
            'title': title,
            'link_url': None,
            'mysql_table_name_subtitle': f"[{config_entry.get('mysql_table_name', config_key)}]",
            'mysql_table_name_raw': config_entry.get('mysql_table_name', config_key),
            'headers': headers,
            'rows': rows
        })
     
    return processed_tables

def prepare_summary_table_data(all_processed_data: dict, sankey_available_years: list, max_years: int = 10):
    summary_data = {'UGP': [], 'UKGP': []}
    if not sankey_available_years: return summary_data

    relevant_years = sorted(sankey_available_years, reverse=True)[:max_years]

    for report_type in ['UGP', 'UKGP']:
        for year in relevant_years:
            year_data = all_processed_data.get(year, {}).get(report_type)
            
            if not isinstance(year_data, dict):
                continue

            income = year_data.get('income_data') or {}
            balance = year_data.get('balance_data') or {}
            fs_data = year_data.get('fs_data') or {}

            factor_text = fs_data.get('rounded_to_nearest', 'ONES')
            factor = 1000 if factor_text == 'THOUSANDS' else (1000000 if factor_text == 'MILLIONS' else 1)

            turnover = get_raw_value(income, 'net_turnover')
            profit = get_raw_value(income, 'net_income')
            assets = get_raw_value(balance, 'total_assets')

            summary_data[report_type].append({
                'year': year,
                'turnover': turnover * factor if turnover is not None else None,
                'profit': profit * factor if profit is not None else None,
                'assets': assets * factor if assets is not None else None,
                'employees': get_raw_value(fs_data, 'employees'),
                'currency': fs_data.get('currency', 'EUR')
            })
    return summary_data

def prepare_ratios_history_for_charts(all_processed_data: dict, max_years: int = 5):
    if not all_processed_data:
        return {}

    available_years = sorted([year for year in all_processed_data if all_processed_data[year].get('UGP')], reverse=True)
    years_to_process = available_years[:max_years]

    ratios_history = {}

    for year in reversed(years_to_process):
        year_data = all_processed_data.get(year, {}).get('UGP', {})
        ratios = year_data.get('financial_ratios', {})
        
        if not ratios:
            continue

        for ratio_key, ratio_values in ratios.items():
            if ratio_key not in ratios_history:
                ratios_history[ratio_key] = []
            
            value = ratio_values.get('value')
            if isinstance(value, (int, float)) and any(s in ratio_key for s in ['margin', 'roa', 'roe', 'roce']):
                value *= 100

            if value is not None:
                 ratios_history[ratio_key].append({
                    'year': str(year),
                    'value': value
                })

    return ratios_history

def build_js_config(page_data: dict):
    ratios_history_data = prepare_ratios_history_for_charts(page_data.get('allProcessedData', {}))

    js_config = {
        'minAutocompleteChars': MIN_AUTOCOMPLETE_CHARS,
        'maxSearchTermLength': MAX_SEARCH_TERM_LENGTH,
        'isLocalEnv': False,
        'baseActionUrl': "#",
        'assetsBasePath': ASSETS_BASE_PATH_FOR_HTML,
        'allProcessedData': page_data.get('allProcessedData', {}),
        'sankeyAvailableYears': page_data.get('sankeyAvailableYears', []),
        'dataAvailableForCharts': bool(page_data.get('sankeyAvailableYears')),
        'searchRegNr': page_data.get('search_reg_nr'),
        'naceCode': page_data.get('nace_code'),
        'summaryTableData': page_data.get('summary_table_data_for_js', {'UGP': [], 'UKGP': []}),
        'ratiosHistory': ratios_history_data
    }
    sanitized_config = sanitize_for_json(js_config)
    return json.dumps(sanitized_config, ensure_ascii=False, allow_nan=False)

def get_company_nace_info(results: dict, nace_df: pd.DataFrame) -> dict:
    # MODIFIKĀCIJA: Noklusējuma vērtības iestatītas uz virtuālo grupu
    nace_code_str = '0000'
    nace_description = 'Nenoteikta nozare'
    
    try:
        tax_data = results.get('pdb_nm_komersantu_samaksato_nodoklu_kopsumas_odata', [])

        if tax_data:
            for row in tax_data:
                row['Taksacijas_gads_num'] = pd.to_numeric(row.get('Taksacijas_gads'), errors='coerce')
            
            # Kārtojam, lai jaunākais gads ir pirmais
            sorted_tax_data = sorted([row for row in tax_data if pd.notna(row['Taksacijas_gads_num'])], 
                                     key=lambda x: x['Taksacijas_gads_num'], 
                                     reverse=True)
            
            for record in sorted_tax_data:
                raw_code = get_raw_value(record, 'Pamatdarbibas_NACE_kods')
                # Ignorējam nederīgus kodus
                if raw_code and str(raw_code).strip() not in ['?', '', '0', '00', 'nan', 'None']:
                    candidate = str(raw_code).strip()
                    # Ja tas nav '0000', mēs to pieņemam kā īsto
                    if candidate != '0000':
                        nace_code_str = candidate
                        # Reset description to None so we look it up freshly
                        nace_description = None 
                        break

        # Ja kods atrasts (un nav 0000), meklējam aprakstu
        if nace_code_str != '0000' and not nace_df.empty:
            match = nace_df[nace_df['Kods'] == nace_code_str]
            if not match.empty:
                nace_description = match.iloc[0]['Nosaukums']
            # Ja kods netika atrasts NACE tabulā, nace_description paliek None
            # Frontendā tas varētu netikt parādīts, bet kods būs.

    except Exception as e:
        print(f"Kļūda, nosakot NACE informāciju: {e}")

    return {'nace_code': nace_code_str, 'nace_description': nace_description}