# projects/python_generator/processing/financial_engine.py

import pandas as pd
from .formatters import get_raw_value, prepare_for_chart
import math

def calculate_financial_ratios(income_data: pd.Series, balance_data: pd.Series):
    """
    Aprēķina galvenos finanšu un riska rādītājus.
    Atgriež vārdnīcu ar aprēķinātajām vērtībām, formulām un izejas datiem.
    """
    if not isinstance(income_data, pd.Series) or not isinstance(balance_data, pd.Series):
        return {}

    # Helper funkcija drošai vērtību iegūšanai
    def get_val(series, key):
        return prepare_for_chart(get_raw_value(series, key)) or 0

    # Datu iegūšana no sērijām
    i = {k: get_val(income_data, k) for k in income_data.index}
    b = {k: get_val(balance_data, k) for k in balance_data.index}

    # Rezultātu vārdnīca
    ratios = {}

    # === Aprēķinu loģika ===
    def _calculate_ratio(name, formula_str, calculation_logic, components):
        try:
            value = calculation_logic()
            if pd.isna(value) or not pd.api.types.is_number(value) or not math.isfinite(value):
                value = None
        except (ZeroDivisionError, TypeError):
            value = None

        formula_parts = formula_str.split(' / ')
        if len(formula_parts) == 2:
            num, den = formula_parts
            formula_html = f'<math display="inline" style="font-size: 1.1em;"><mfrac><mrow><mtext>{num}</mtext></mrow><mrow><mtext>{den}</mtext></mrow></mfrac></math>'
        else:
            formula_html = formula_str

        if value is not None and len(components) == 2 and all(isinstance(c, (int, float)) for c in components):
            comp_num, comp_den = components
            # Format with spaces as thousand separators
            formatted_num = f"{comp_num:,.2f}".replace(",", " ")
            formatted_den = f"{comp_den:,.2f}".replace(",", " ")
            calculation_html = f'<math display="inline" style="font-size: 1.1em;"><mfrac><mrow><mtext>{formatted_num}</mtext></mrow><mrow><mtext>{formatted_den}</mtext></mrow></mfrac></math>'
        else:
            calculation_html = "aprēķins nav iespējams"

        ratios[name] = {
            'value': value,
            'formula': formula_html,
            'calculation': calculation_html,
            'components': {k: v for k, v in zip(['A', 'B', 'C'], components)}
        }


    # 1. Likviditātes rādītāji
    _calculate_ratio('current_ratio', "Apgrozāmie Līdzekļi / Īstermiņa Saistības",
                     lambda: b['total_current_assets'] / b['current_liabilities'],
                     [b['total_current_assets'], b['current_liabilities']])
    
    quick_assets = b['total_current_assets'] - b['inventories']
    _calculate_ratio('quick_ratio', "(Apgrozāmie Līdzekļi - Krājumi) / Īstermiņa Saistības",
                     lambda: quick_assets / b['current_liabilities'],
                     [quick_assets, b['current_liabilities']])

    # 2. Rentabilitātes rādītāji
    _calculate_ratio('net_profit_margin', "Tīrā Peļņa / Neto Apgrozījums",
                     lambda: i['net_income'] / i['net_turnover'],
                     [i['net_income'], i['net_turnover']])
    
    _calculate_ratio('roa', "Tīrā Peļņa / Aktīvi Kopā",
                     lambda: i['net_income'] / b['total_assets'],
                     [i['net_income'], b['total_assets']])

    _calculate_ratio('roe', "Tīrā Peļņa / Pašu Kapitāls",
                     lambda: i['net_income'] / b['equity'],
                     [i['net_income'], b['equity']])

    # 3. Maksātspējas rādītāji
    total_liabilities = b['current_liabilities'] + b['non_current_liabilities']
    _calculate_ratio('debt_to_equity', "Saistības Kopā / Pašu Kapitāls",
                     lambda: total_liabilities / b['equity'],
                     [total_liabilities, b['equity']])
    
    ebit = i['income_before_income_taxes'] + i['interest_expenses']
    _calculate_ratio('interest_coverage', "EBIT / Procentu Izmaksas",
                     lambda: ebit / i['interest_expenses'],
                     [ebit, i['interest_expenses']])

    # 4. Efektivitātes rādītāji
    _calculate_ratio('asset_turnover', "Neto Apgrozījums / Aktīvi Kopā",
                     lambda: i['net_turnover'] / b['total_assets'],
                     [i['net_turnover'], b['total_assets']])

    # 5. Maksātnespējas risks (Altman Z-score)
    try:
        if b['total_assets'] > 0:
            A = (b['total_current_assets'] - b['current_liabilities']) / b['total_assets'] # Working Capital / Total Assets (X1)
            B = b['equity'] / b['total_assets'] # Retained Earnings (Equity as proxy) / Total Assets (X2)
            C = ebit / b['total_assets'] # EBIT / Total Assets (X3)
            D = b['equity'] / total_liabilities if total_liabilities > 0 else 0 # Book Value of Equity / Total Liabilities (X4)
            E = i['net_turnover'] / b['total_assets'] # Sales / Total Assets (X5)
            
            # KORIĢĒTS: Altman Z'-Score (1983) modelis privātiem uzņēmumiem
            z_score = 0.717 * A + 0.847 * B + 3.107 * C + 0.420 * D + 0.998 * E
            
            z_calc_str = (f"0.717*{A:.2f} + 0.847*{B:.2f} + 3.107*{C:.2f} + 0.420*{D:.2f} + 0.998*{E:.2f}")

            ratios['altman_z_score'] = {
                'value': z_score,
                'formula': "Z' (1983) = 0.717*A + 0.847*B + 3.107*C + 0.420*D + 0.998*E",
                'calculation': z_calc_str,
                'components': {'A': A, 'B': B, 'C': C, 'D': D, 'E': E }
            }
        else:
            raise ZeroDivisionError
    except (ZeroDivisionError, TypeError):
        ratios['altman_z_score'] = {'value': None, 'formula': "Z aprēķins nav iespējams", 'calculation': 'Nav datu'}

    # JAUNS: 6. Kapitāla efektivitāte (ROCE)
    iegulditais_kapitals = b['total_assets'] - b['current_liabilities']
    _calculate_ratio('roce', "EBIT / Ieguldītais Kapitāls",
                     lambda: ebit / iegulditais_kapitals,
                     [ebit, iegulditais_kapitals])
    
    return ratios

def prepare_waterfall_sankey_data(income_data_row_series: pd.Series, rounding_multiplier: int = 1):
    """
    Sagatavo datus D3.js Sankey diagrammai, veidojot "ūdenskrituma" plūsmu.
    Automātiski izvēlas detalizētāko pieejamo datu formātu (pēc funkcijas vai pēc būtības)
    un pielieto noapaļošanas reizinātāju.
    """
    if not isinstance(income_data_row_series, pd.Series) or income_data_row_series.empty:
        return None

    def get_val(key):
        raw_val = prepare_for_chart(income_data_row_series.get(key)) or 0
        return raw_val * rounding_multiplier

    use_by_function_method = get_val('by_function_cost_of_goods_sold') != 0 or get_val('by_function_gross_profit') != 0
    nodes, links, nodes_added = [], [], set()

    def add_node_if_new(node_id, node_name):
        if node_id not in nodes_added:
            nodes.append({"id": node_id, "name": node_name})
            nodes_added.add(node_id)

    def add_link(source_id, target_id, value, node_map):
        if value == 0: return
        add_node_if_new(source_id, node_map.get(source_id, source_id))
        add_node_if_new(target_id, node_map.get(target_id, target_id))
        links.append({
            "source": source_id, "target": target_id,
            "value": abs(round(value)), "originalValue": round(value),
            "isLoss": value < 0
        })

    net_income_final = get_val('net_income')

    if use_by_function_method:
        net_turnover, cogs, gross_profit = get_val('net_turnover'), get_val('by_function_cost_of_goods_sold'), get_val('by_function_gross_profit')
        if gross_profit == 0 and (net_turnover != 0 or cogs != 0):
            gross_profit = net_turnover - cogs
        selling_exp, admin_exp, other_op_rev, other_op_exp = get_val('by_function_selling_expenses'), get_val('by_function_administrative_expenses'), get_val('by_function_other_operating_revenues'), get_val('other_operating_expenses')
        interest_revenue, interest_exp, tax = get_val('other_interest_revenues'), get_val('interest_expenses'), get_val('provision_for_income_taxes')
        ebit = gross_profit - selling_exp - admin_exp - other_op_exp + other_op_rev
        node_map = {"net_turnover_source": "Neto Apgrozījums", "gross_profit_calc": "Bruto Peļņa", "cogs_exp": "Pārdotās prod. pašizmaksa", "selling_exp": "Pārdošanas izmaksas", "admin_exp": "Administratīvās izmaksas", "other_op_rev_source": "Pārējie saimn. darb. ieņēmumi", "other_op_exp_exp": "Pārējās saimn. darb. izmaksas", "ebit_calc": "Peļņa pirms proc. un nodokļiem (EBIT)", "interest_rev_source": "Procentu Ieņēmumi", "interest_exp": "Procentu Izmaksas", "ebt_calc": "Peļņa pirms nodokļiem (EBT)", "tax_exp": "UIN Nodoklis", "net_income_final": f"Tīrā Peļņa{' (Zaudējumi)' if net_income_final < 0 else ''}"}
        add_link("net_turnover_source", "cogs_exp", cogs, node_map)
        add_link("net_turnover_source", "gross_profit_calc", gross_profit, node_map)
        add_link("gross_profit_calc", "selling_exp", selling_exp, node_map)
        add_link("gross_profit_calc", "admin_exp", admin_exp, node_map)
        add_link("gross_profit_calc", "other_op_exp_exp", other_op_exp, node_map)
        add_link("gross_profit_calc", "ebit_calc", ebit, node_map)
        if other_op_rev != 0: add_link("other_op_rev_source", "ebit_calc", other_op_rev, node_map)
        add_link("ebit_calc", "ebt_calc", ebit, node_map)
        if interest_revenue != 0: add_link("interest_rev_source", "ebt_calc", interest_revenue, node_map)
        add_link("ebt_calc", "interest_exp", interest_exp, node_map)
        add_link("ebt_calc", "tax_exp", tax, node_map)
        add_link("ebt_calc", "net_income_final", net_income_final, node_map)
    else:
        net_turnover, other_op_revenue, inventory_change = get_val('net_turnover'), get_val('by_nature_other_operating_revenues'), get_val('by_nature_inventory_change')
        material_costs, labour_costs, other_op_exp, depreciation = get_val('by_nature_material_expenses'), get_val('by_nature_labour_expenses'), get_val('other_operating_expenses'), get_val('by_nature_depreciation_expenses')
        interest_revenue, interest_exp, tax = get_val('other_interest_revenues'), get_val('interest_expenses'), get_val('provision_for_income_taxes')
        total_op_revenue, total_op_costs = net_turnover + other_op_revenue + inventory_change, material_costs + labour_costs + other_op_exp
        ebitda, ebit = total_op_revenue - total_op_costs, total_op_revenue - total_op_costs - depreciation
        node_map = {"net_turnover_source": "Neto Apgrozījums", "other_rev_source": "Pārējie Ieņēmumi & Krāj. Izm.", "total_rev_calc": "Kopējie Darb. Ieņēmumi", "material_costs_exp": "Materiālu Izmaksas", "labour_costs_exp": "Personāla Izmaksas", "other_op_costs_exp": "Pārējās Operat. Izmaksas", "ebitda_calc": "EBITDA", "depreciation_exp": "Amortizācija (D&A)", "ebit_calc": "EBIT", "interest_rev_source": "Procentu Ieņēmumi", "interest_exp": "Procentu Izmaksas", "ebt_calc": "EBT", "tax_exp": "UIN Nodoklis", "net_income_final": f"Tīrā Peļņa{' (Zaudējumi)' if net_income_final < 0 else ''}"}
        add_link("net_turnover_source", "total_rev_calc", net_turnover, node_map)
        add_link("other_rev_source", "total_rev_calc", other_op_revenue + inventory_change, node_map)
        add_link("total_rev_calc", "material_costs_exp", material_costs, node_map)
        add_link("total_rev_calc", "labour_costs_exp", labour_costs, node_map)
        add_link("total_rev_calc", "other_op_costs_exp", other_op_exp, node_map)
        add_link("total_rev_calc", "ebitda_calc", ebitda, node_map)
        add_link("ebitda_calc", "depreciation_exp", depreciation, node_map)
        add_link("ebitda_calc", "ebit_calc", ebit, node_map)
        add_link("ebit_calc", "ebt_calc", ebit, node_map)
        if interest_revenue != 0: add_link("interest_rev_source", "ebt_calc", interest_revenue, node_map)
        add_link("ebt_calc", "interest_exp", interest_exp, node_map)
        add_link("ebt_calc", "tax_exp", tax, node_map)
        add_link("ebt_calc", "net_income_final", net_income_final, node_map)

    return {"nodes": nodes, "links": links}

def get_financial_statements_info(df_financial_statements: pd.DataFrame, reg_nr_str: str):
    if df_financial_statements is None or df_financial_statements.empty: return {}, []
    company_fs = df_financial_statements[df_financial_statements['legal_entity_registration_number'] == str(reg_nr_str)].copy()
    if company_fs.empty: return {}, []
    company_fs['year'] = pd.to_numeric(company_fs['year'], errors='coerce')
    company_fs.dropna(subset=['year'], inplace=True)
    if company_fs.empty: return {}, []
    company_fs['year'] = company_fs['year'].astype(int)
    company_fs_sorted = company_fs.sort_values(by='year', ascending=True)
    statements_info_dict, statement_ids_list_result = {}, []
    for _, row_series in company_fs_sorted.iterrows():
        sid_val = get_raw_value(row_series, 'id')
        if sid_val is not None:
            sid_str = str(sid_val)
            if sid_str not in statements_info_dict:
                statements_info_dict[sid_str] = row_series.to_dict()
                statement_ids_list_result.append(sid_str)
    return statements_info_dict, statement_ids_list_result

def get_financial_details_data(dataframes: dict, fs_ids_list: list):
    income_data, balance_data, cash_flow_data = {}, {}, {}
    if not fs_ids_list: return income_data, balance_data, cash_flow_data
    unique_fs_ids = list(set(fs_ids_list))
    for table_name, data_dict in [('income_statements', income_data), ('balance_sheets', balance_data), ('cash_flow_statements', cash_flow_data)]:
        df = dataframes.get(table_name)
        if df is not None and not df.empty and 'statement_id' in df.columns:
            subset = df[df['statement_id'].isin(unique_fs_ids)].drop_duplicates(subset=['statement_id'], keep='first')
            for _, row in subset.iterrows():
                data_dict[row['statement_id']] = row
    return income_data, balance_data, cash_flow_data

def process_financial_data_for_years(statements_info: dict, income_data: dict, balance_data: dict, cash_flow_data: dict):
    all_processed_data = {}
    if not statements_info: return {}, []

    valid_fs_ids = [fs_id for fs_id in statements_info if isinstance(statements_info.get(fs_id), dict)]
    sorted_fs_ids = sorted(valid_fs_ids, key=lambda fs_id: statements_info[fs_id].get('year', 0))

    for fs_id in sorted_fs_ids:
        info_entry = statements_info.get(fs_id)
        if not info_entry: continue

        year = info_entry.get('year')
        if year is None: continue

        current_income = income_data.get(fs_id)
        current_balance = balance_data.get(fs_id)
        current_cash_flow = cash_flow_data.get(fs_id)

        if not isinstance(current_income, pd.Series) and not isinstance(current_balance, pd.Series):
            continue

        rounding_multiplier = 1
        rounded_to_str = str(info_entry.get('rounded_to_nearest', 'ONES')).strip().upper()
        if rounded_to_str == 'THOUSANDS': rounding_multiplier = 1000
        elif rounded_to_str == 'MILLIONS': rounding_multiplier = 1000000

        report_type = str(info_entry.get('source_type', 'UGP')).strip().upper()
        
        year_data_output = {'currency': str(info_entry.get('currency', 'EUR')).strip(), 'rounded_to_nearest': str(info_entry.get('rounded_to_nearest', 'ONES')).strip().upper(), 'employees': get_raw_value(info_entry, 'employees')}
        
        source_ids = {'financial_statement_id': fs_id, 'income_statement_id': get_raw_value(current_income, 'id') or get_raw_value(current_income, 'statement_id'), 'balance_sheet_id': get_raw_value(current_balance, 'id') or get_raw_value(current_balance, 'statement_id'), 'cash_flow_statement_id': get_raw_value(current_cash_flow, 'id') or get_raw_value(current_cash_flow, 'statement_id')}
        for k, v in source_ids.items():
            if v is not None: source_ids[k] = str(v)

        if year not in all_processed_data: all_processed_data[year] = {}
        valid_type_key = 'UGP' if report_type != 'UKGP' else 'UKGP'

        sankey_prepared_data = None
        if isinstance(current_income, pd.Series):
            sankey_prepared_data = prepare_waterfall_sankey_data(current_income, rounding_multiplier)

        # Aprēķinām finanšu rādītājus un pievienojam tos datiem
        financial_ratios = calculate_financial_ratios(current_income, current_balance)

        all_processed_data[year][valid_type_key] = {
            'fs_data': {**year_data_output, 'type_from_db': report_type if pd.notna(report_type) else ''},
            'income_data': current_income.to_dict() if isinstance(current_income, pd.Series) else None,
            'balance_data': current_balance.to_dict() if isinstance(current_balance, pd.Series) else None,
            'cash_flow_data': current_cash_flow.to_dict() if isinstance(current_cash_flow, pd.Series) else None,
            'sankey_prepared': sankey_prepared_data,
            'financial_ratios': financial_ratios,
            'source_ids': source_ids
        }

    sorted_all_processed_data = dict(sorted(all_processed_data.items()))
    sankey_available_years = []

    for year, types_data in sorted_all_processed_data.items():
        ugp_data = types_data.get('UGP')
        ugp_has_links = False
        if isinstance(ugp_data, dict):
            sankey_prepared = ugp_data.get('sankey_prepared')
            if isinstance(sankey_prepared, dict):
                if sankey_prepared.get('links'):
                    ugp_has_links = True

        ukgp_data = types_data.get('UKGP')
        ukgp_has_links = False
        if isinstance(ukgp_data, dict):
            sankey_prepared = ukgp_data.get('sankey_prepared')
            if isinstance(sankey_prepared, dict):
                if sankey_prepared.get('links'):
                    ukgp_has_links = True
        
        if ugp_has_links or ukgp_has_links:
            sankey_available_years.append(year)

    return sorted_all_processed_data, sorted(sankey_available_years)