# kods/py/file_ops.py

import shutil
import os
import requests
from pathlib import Path
from .paths import KODS_DIR, OUT_DIR, TEMPLATES_DIR, COMPANIES_DB

# Jaunais ceļš uz NACE DB (kas atrodas kods/nace_stats.sqlite pēc 1. soļa)
NACE_DB = KODS_DIR / "nace_stats.sqlite"

def sync_resources():
    """
    Kopē visus nepieciešamos failus un mapes uz 'out' direktoriju.
    """
    print("   -> Sinhronizē resursus...")
    
    if not OUT_DIR.exists():
        OUT_DIR.mkdir(parents=True, exist_ok=True)
    
    # 1. Moduļu mapes (Head, Footer, Cookie)
    # 'pozicija' mapi vairs nevajag, jo tagad lietojam SQLite DB
    folders_to_sync = ['head', 'footer', 'cookie']
    
    for folder in folders_to_sync:
        src = KODS_DIR / folder
        dest = OUT_DIR / folder
        
        if src.exists():
            if dest.exists(): shutil.rmtree(dest)
            shutil.copytree(src, dest)
        else:
            print(f"⚠️  BRĪDINĀJUMS: Modulis '{folder}' nav atrasts iekš '{KODS_DIR}'")

    # 2. Assets mape (CSS, JS, IMG)
    assets_src = TEMPLATES_DIR / "assets"
    assets_dest = OUT_DIR / "assets"
    
    if assets_src.exists():
        if assets_dest.exists(): shutil.rmtree(assets_dest)
        shutil.copytree(assets_src, assets_dest)
    else:
        print(f"⚠️  BRĪDINĀJUMS: Assets mape nav atrasta: {assets_src}")

    # 3. Meklēšanas un Statistikas faili (Backend)
    # Mēs visu liekam assets/search mapē, lai būtu vienuviet
    search_dir = assets_dest / "search"
    search_dir.mkdir(parents=True, exist_ok=True)
    
    # A) Meklēšanas DB (companies.sqlite)
    if COMPANIES_DB.exists():
        shutil.copy2(COMPANIES_DB, search_dir / "companies.sqlite")
    else:
        (search_dir / "companies.sqlite").touch() # Tukšs fails, lai nebūtu kļūdu
        print("ℹ️  Izveidots tukšs companies.sqlite")

    # B) NACE Statistikas DB (nace_stats.sqlite) - JAUNAIS
    if NACE_DB.exists():
        shutil.copy2(NACE_DB, search_dir / "nace_stats.sqlite")
        print("      + NACE statistikas DB nokopēta")
    else:
        print("⚠️  NACE DB nav atrasta. Vai palaidāt 1_prepare_data.py?")

    # C) PHP Skripti (Search API un NACE API)
    php_src_dir = TEMPLATES_DIR / "php_backend"
    
    # search.php
    if (php_src_dir / "search.php").exists():
        shutil.copy2(php_src_dir / "search.php", search_dir / "search.php")
        
    # nace_stats.php - JAUNAIS
    if (php_src_dir / "nace_stats.php").exists():
        shutil.copy2(php_src_dir / "nace_stats.php", search_dir / "nace_stats.php")
    else:
        print("⚠️  nace_stats.php nav atrasts php_backend mapē.")

    # 4. Header.php (ja tas ir atsevišķi)
    if (TEMPLATES_DIR / "partials" / "header.php").exists():
        shutil.copy2(TEMPLATES_DIR / "partials" / "header.php", OUT_DIR / "header.php")

def download_js_libs():
    """
    Lejupielādē ārējās JS bibliotēkas lokālai lietošanai (lai neizmantotu CDN).
    """
    lib_dir = OUT_DIR / "assets" / "js" / "lib"
    lib_dir.mkdir(parents=True, exist_ok=True)
    
    libs = {
        "chart.umd.min.js": "https://cdn.jsdelivr.net/npm/chart.js@4.4.1/dist/chart.umd.min.js",
        "d3.min.js": "https://cdnjs.cloudflare.com/ajax/libs/d3/7.9.0/d3.min.js",
        "d3-sankey.min.js": "https://cdnjs.cloudflare.com/ajax/libs/d3-sankey/0.12.3/d3-sankey.min.js"
    }
    
    for fname, url in libs.items():
        fpath = lib_dir / fname
        if not fpath.exists():
            print(f"   -> Lejupielādē: {fname}...")
            try:
                r = requests.get(url, timeout=10)
                r.raise_for_status()
                with open(fpath, 'wb') as f: f.write(r.content)
            except Exception as e: 
                print(f"⚠️ Kļūda lejupielādējot {fname}: {e}")