# kods/py/renderer.py
import pandas as pd
import pickle
import gc
import random
import time
import traceback
import sys
from datetime import datetime
from jinja2 import Environment, FileSystemLoader, select_autoescape
from pathlib import Path

# Iestatām ceļus, lai atrastu mūsu moduļus
from .paths import KODS_DIR, OUT_DIR, TEMPLATES_DIR

# Mums jāpievieno KODS_DIR sistēmas ceļam, lai importētu 'processing', 'core', etc.
if str(KODS_DIR) not in sys.path:
    sys.path.append(str(KODS_DIR))

from config import settings
from processing import financial_engine, formatters, segmenter
from core import data_fetcher, page_builder
from templates.TXT.descriptions import descriptions, financial_descriptions, DESCRIPTION_DISCLAIMER

# Globālais konteksts (katram procesam savs)
worker_context = {}

def init_worker(pickle_path):
    """Ielādē datus un Jinja2 vidi katrā procesā."""
    global worker_context
    
    # 1. Datu ielāde (Mac Optimizācija)
    try:
        with open(pickle_path, 'rb') as f:
            data = pickle.load(f)
            worker_context['dfs'] = data['dataframes']
            worker_context['nace'] = data['nace_df']
    except Exception as e:
        print(f"Worker Init Error: {e}")
        return

    # 2. Jinja2
    env = Environment(
        loader=FileSystemLoader(TEMPLATES_DIR),
        autoescape=select_autoescape(['html', 'xml', 'php']),
        trim_blocks=True, lstrip_blocks=True
    )
    env.filters['format_number'] = lambda val, cur='EUR', factor=1: formatters.format_number_data(val, cur, factor)
    env.filters['format_plain_number'] = lambda val: f"{int(val):,}".replace(",", " ") if val is not None and isinstance(val, (int, float)) and val == int(val) else (f"{val:,.0f}".replace(",", " ") if val is not None and isinstance(val, (int, float)) else ("—" if val is None else val))
    
    worker_context['env'] = env
    gc.collect()

# --- Palīgfunkcijas ---
def generate_desc(date_str):
    try:
        d = pd.to_datetime(date_str, format='%Y-%m-%d', errors='coerce')
        age = datetime.now().year - d.year
        for (mn, mx), ph in descriptions.items():
            if mn <= age <= mx: return random.choice(ph)
    except: pass
    return ""

def get_ratio_sent(val, th, sent):
    if val is None: return None
    color_map = {
        'danger': 'red', 'distress': 'red', 'high_risk': 'red', 'low': 'red', 
        'precaution': 'orange', 'grey': 'orange', 'borderline': 'orange', 'elevated_risk': 'orange', 'average': 'orange', 'satisfactory': 'orange',
        'healthy': 'green', 'safe': 'green', 'low_risk': 'green', 'high': 'green', 
        'excessive': 'blue', 'inefficient': 'blue'
    }
    cat = None
    
    if 'distress' in th: cat = 'distress' if val < th['distress'] else ('grey' if val <= th['grey'] else 'safe')
    elif 'four_level' in th:
        k = list(sent.keys())
        cat = k[0] if val < th['level1_max'] else (k[1] if val < th['level2_max'] else (k[2] if val < th['level3_max'] else k[3]))
    elif th.get('invert_logic'):
        k = list(sent.keys())
        cat = k[0] if val < th['low'] else (k[1] if val <= th['high'] else k[2])
    else:
        k = list(sent.keys())
        cat = k[0] if val < th['low'] else (k[1] if val <= th['high'] else k[2])
        
    if cat and cat in sent:
        s = random.choice(sent[cat])
        return f'<span style="color: {color_map.get(cat, "black")};">{s.replace("<br>", "")}</span><br>'
    return None

def process_company(reg_nr):
    """Ģenerē vienu uzņēmuma lapu."""
    dfs = worker_context['dfs']
    env = worker_context['env']
    nace_df = worker_context['nace']

    try:
        # Datu savākšana
        main = data_fetcher.fetch_main_company_data(dfs.get('register'), reg_nr)
        all_res = data_fetcher.fetch_all_data_for_reg_nr(dfs, reg_nr, list(dfs.keys()))
        memb_ent = data_fetcher.fetch_member_as_entity_data(dfs.get('members'), reg_nr)
        
        # Finanses
        fs_info, fs_ids = financial_engine.get_financial_statements_info(dfs.get('financial_statements'), reg_nr)
        inc, bal, cash = financial_engine.get_financial_details_data(dfs, fs_ids)
        proc_fin, s_years = financial_engine.process_financial_data_for_years(fs_info, inc, bal, cash)
        summary = page_builder.prepare_summary_table_data(proc_fin, s_years)
        
        segment = segmenter.determine_company_segment(main, summary)
        
        # Apraksts
        desc_list = []
        age_d = generate_desc(main.get('registered'))
        if age_d:
            n = str(main.get('name_in_quotes', '')).strip()
            desc_list.append(f'{str(main.get("type", "")).strip()} "{n}" ir {age_d}' if n else age_d)

        # Finanšu rādītāju teksts
        try:
            latest = None
            ly = None # Inicializējam gadu
            if s_years:
                ly = max(s_years)
                if proc_fin.get(ly):
                    latest = proc_fin[ly].get('UGP') or proc_fin[ly].get('UKGP')
            
            if latest and latest.get('financial_ratios'):
                ratios = latest['financial_ratios']
                
                # --- LABOJUMS: Pievienots teikuma ievads ---
                if ly:
                    desc_list.append(f"Pamatojoties uz {ly}. gada datiem,")
                # -------------------------------------------

                # Definējam pilnu konfigurāciju
                defs = {
                    'current_ratio': {'thresholds': {'four_level': True, 'level1_max': 1.0, 'level2_max': 1.5, 'level3_max': 3.0}},
                    'quick_ratio': {'thresholds': {'four_level': True, 'level1_max': 0.8, 'level2_max': 1.0, 'level3_max': 2.0}},
                    'net_profit_margin': {'thresholds': {'low': 0, 'high': 10}},
                    'roa': {'thresholds': {'low': 2, 'high': 5}},
                    'roe': {'thresholds': {'low': 5, 'high': 15}},
                    'debt_to_equity': {'thresholds': {'low': 1.5, 'high': 2.0, 'invert_logic': True}},
                    'interest_coverage': {'thresholds': {'low': 1.5, 'high': 3.0}},
                    'asset_turnover': {'thresholds': {'low': 0.5, 'high': 1.0}},
                    'roce': {'thresholds': {'low': 5, 'high': 15}},
                    'altman_z_score': {'thresholds': {'distress': 1.23, 'grey': 2.90}}
                }

                # Iterējam cauri un ģenerējam teikumus
                for name, definition in defs.items():
                    ratio_data = ratios.get(name)
                    if ratio_data:
                        ratio_value = ratio_data.get('value')
                        
                        # Procentuālajiem rādītājiem reizinām ar 100
                        if name in ['net_profit_margin', 'roa', 'roe', 'roce']:
                            if ratio_value is not None: ratio_value *= 100
                        
                        sentences = financial_descriptions.get(name)
                        if sentences:
                            sentence = get_ratio_sent(ratio_value, definition['thresholds'], sentences)
                            if sentence: desc_list.append(sentence)
            
        except Exception as e:
            pass

        full_desc = " ".join(desc_list) + f"<br><br>{DESCRIPTION_DISCLAIMER}"

        # Datu Paka
        p_data = {
            'search_reg_nr': reg_nr,
            'dati_php_rowData': main.to_dict() if main is not None else {},
            'results': all_res,
            'member_as_entity_data': memb_ent,
            'allProcessedData': proc_fin,
            'sankeyAvailableYears': s_years,
            'errors': [],
            'segment': segment,
            'summary_table_data_for_js': summary,
            'company_description': full_desc,
        }

        panel = page_builder.get_company_details_for_panel(main, reg_nr, segment)
        nace = page_builder.get_company_nace_info(all_res, nace_df)
        vid = page_builder.prepare_vid_panel_data(all_res)

        panel.update(nace)
        p_data.update(panel)
        p_data['vid_panel_data'] = vid
        p_data['seo_metadata'] = page_builder.prepare_seo_metadata(p_data)
        p_data['search_term_value'] = panel['companyTitleForHtml']
        p_data['current_page_title'] = f"{panel['companyTitleForHtml']} ({reg_nr})"
        p_data['processed_tables_for_display'] = page_builder.prepare_data_for_results_tables(p_data)

        # SEO Links
        h_links = set()
        if sup_id := data_fetcher.fetch_super_id(dfs.get('register'), reg_nr):
             t_ids = [sup_id + o for o in [1, -1, 200, -200] if sup_id + o > 0]
             h_links.update(data_fetcher.fetch_regcodes_by_super_ids(dfs.get('register'), t_ids, active_only=True))
        h_links.update(data_fetcher.fetch_random_regcodes(dfs.get('register'), 5, active_only=True))

        p_data['has_summary_data'] = bool(summary.get('UGP') or summary.get('UKGP'))
        p_data['has_financial_charts'] = bool(s_years)
        p_data['generated_hidden_links'] = [f"{settings.BASE_DOMAIN}/{rc}" for rc in sorted(h_links) if rc != reg_nr]
        p_data['canonical_url'] = f"{settings.BASE_DOMAIN}/{reg_nr}"
        p_data['js_config_json'] = page_builder.build_js_config(p_data)
        p_data['generationDate'] = datetime.now().strftime('%d.%m.%Y')
        p_data['base_project_url'] = settings.BASE_PROJECT_URL_FOR_HTML_LINKS
        p_data['assets_base_path'] = "/assets"

        # Renderēšana un Saglabāšana
        html = env.get_template('layout.html').render(page_data=p_data)
        
        d1, d2 = reg_nr[0:2], reg_nr[2:4]
        target_dir = OUT_DIR / "x" / d1 / d2
        target_dir.mkdir(parents=True, exist_ok=True)
        with open(target_dir / f"{reg_nr}.php", 'w', encoding='utf-8') as f: f.write(html)

        return reg_nr
    except:
        return None

def render_special_pages():
    """Ģenerē index un 404."""
    env = Environment(loader=FileSystemLoader(TEMPLATES_DIR), autoescape=select_autoescape(['html', 'xml', 'php']))
    common = {'assets_base_path': "/assets"}
    
    try:
        with open(OUT_DIR / "index.php", 'w', encoding='utf-8') as f:
            f.write(env.get_template('index.html').render(page_data=common))
        with open(OUT_DIR / "404.php", 'w', encoding='utf-8') as f:
            f.write(env.get_template('404.html').render(page_data=common))
        print("   -> Index un 404 lapas izveidotas.")
    except Exception as e: print(f"Kļūda: {e}")