// assets/js/structure_script.js (Izlabota kļūda ar dubultu mainīgo definēšanu)

document.addEventListener('DOMContentLoaded', function () {
    const width = window.innerWidth;
    const height = window.innerHeight - 90;

    // --- DOM ELEMENTI (Definēti vienu reizi šeit) ---
    const statsBox = document.getElementById('stats-box');
    const statsCount = document.getElementById('stats-count');
    const statsProfit = document.getElementById('stats-profit');
    const toggleButton = document.getElementById('stats-toggle-button');
    const tableContainer = document.getElementById('stats-table-container');
    const tableBody = document.getElementById('stats-table-body');
    const searchForm = document.getElementById('search-form');
    const searchInput = document.getElementById('search-input');
    const noticeContainer = document.querySelector('.data-source-notice');
    const noticeToggleButton = document.querySelector('.notice-toggle');

    // --- D3 UZSTĀDĪJUMI ---
    const svg = d3.select("#chart").attr("width", width).attr("height", height);
    const zoom = d3.zoom().on("zoom", (event) => g.attr("transform", event.transform));
    svg.call(zoom);
    const g = svg.append("g");

    // --- DATU SAGATAVOŠANA ---
    const nodeById = new Map(graph_data.nodes.map(node => [node.id, node]));
    const sourceIds = new Set(graph_data.links.map(l => l.source));
    const targetIds = new Set(graph_data.links.map(l => l.target));

    // --- SIMULĀCIJA UN ZĪMĒŠANA ---
    const simulation = d3.forceSimulation(graph_data.nodes)
        .force("link", d3.forceLink(graph_data.links).id(d => d.id).distance(100))
        .force("charge", d3.forceManyBody().strength(-300))
        .force("center", d3.forceCenter(width / 2, height / 2));

    const link = g.append("g").attr("class", "links").selectAll("line").data(graph_data.links).enter().append("line");
    const node = g.append("g").attr("class", "nodes").selectAll("g").data(graph_data.nodes).enter().append("g").attr("class", "node-group").call(drag(simulation));

    node.append("circle").attr("r", 10).attr("fill", d => {
        const isSource = sourceIds.has(d.id);
        const isTarget = targetIds.has(d.id);
        if (isSource && !isTarget) return "#d62728";
        if (isSource && isTarget) return "#2ca02c";
        return "#1f1f1f";
    });

    const labels = node.append("text").attr("class", "label").attr("dy", -15).attr("text-anchor", "middle").style("display", "none");
    labels.append("tspan").attr("class", "name").attr("x", 0).attr("dy", "0em");
    labels.append("tspan").attr("class", "regcode").attr("x", 0).attr("dy", "1.2em").on("click", (event, d) => {
        event.stopPropagation();
        window.open(`https://saraksts.lv/${d.id}`, '_blank');
    });
    labels.append("tspan").attr("class", "profit").attr("x", 0).attr("dy", "1.2em");

    simulation.on("tick", () => {
        link.attr("x1", d => d.source.x).attr("y1", d => d.source.y).attr("x2", d => d.target.x).attr("y2", d => d.target.y);
        node.attr("transform", d => `translate(${d.x},${d.y})`);
    });

    // --- NOTIKUMU APSTRĀDE ---
    svg.on("click", resetHighlight);
    node.on("click", handleNodeClick);
    searchForm.addEventListener('submit', handleSearch);

    if (noticeContainer && noticeToggleButton) {
        noticeToggleButton.addEventListener('click', () => {
            noticeContainer.classList.toggle('expanded');
            const isExpanded = noticeContainer.classList.contains('expanded');
            noticeToggleButton.textContent = isExpanded ? '−' : '+';
        });
    }

    // Liekās mainīgo definīcijas, kas izraisīja kļūdu, ir noņemtas.

    // --- GALVENĀS FUNKCIJAS ---

    function resetHighlight() {
        node.classed("highlighted", false).classed("muted", false);
        link.classed("highlighted", false).classed("muted", false);
        labels.style("display", "none");
        statsBox.classList.add('hidden');
        tableContainer.classList.add('hidden');
        toggleButton.textContent = "Rādīt sarakstu";
    }

    function handleNodeClick(event, d) {
        event.stopPropagation();
        const isHighlighted = d3.select(this).classed("highlighted");
        resetHighlight();
        if (isHighlighted) return;
        const componentNodeIds = findConnectedComponent(d);
        highlightComponent(componentNodeIds);
        const groupNodes = Array.from(componentNodeIds).map(id => nodeById.get(id));
        updateStats(groupNodes);
    }
    
    function handleSearch() {
        const searchTerm = searchInput.value.trim();
        if (!searchTerm || !/^\d{11}$/.test(searchTerm)) {
            alert("Lūdzu, ievadiet derīgu 11 ciparu reģistrācijas numuru.");
            return;
        }
        const targetNodeData = nodeById.get(searchTerm);
        if (!targetNodeData) {
            alert("Uzņēmums ar šādu reģistrācijas numuru nav atrasts šajā vizualizācijā.");
            return;
        }
        const targetNodeElement = node.filter(d => d.id === searchTerm);
        targetNodeElement.dispatch('click');
        const scale = 2;
        const x = targetNodeData.x;
        const y = targetNodeData.y;
        const transform = d3.zoomIdentity.translate(width / 2, height / 2).scale(scale).translate(-x, -y);
        svg.transition().duration(750).call(zoom.transform, transform);
    }

    function findConnectedComponent(startNode) {
        const componentNodeIds = new Set();
        const queue = [startNode];
        const visited = new Set([startNode.id]);
        while (queue.length > 0) {
            const currentNode = queue.shift();
            componentNodeIds.add(currentNode.id);
            graph_data.links.forEach(l => {
                let neighborId = null;
                if (l.source.id === currentNode.id) neighborId = l.target.id;
                else if (l.target.id === currentNode.id) neighborId = l.source.id;
                if (neighborId && !visited.has(neighborId)) {
                    visited.add(neighborId);
                    const neighborNode = nodeById.get(neighborId);
                    if (neighborNode) queue.push(neighborNode);
                }
            });
        }
        return componentNodeIds;
    }

    function highlightComponent(componentNodeIds) {
        node.classed("muted", true);
        link.classed("muted", true);
        node.filter(n => componentNodeIds.has(n.id)).classed("highlighted", true).classed("muted", false);
        link.filter(l => componentNodeIds.has(l.source.id) && componentNodeIds.has(l.target.id)).classed("highlighted", true).classed("muted", false);
        const highlightedLabels = node.filter(n => componentNodeIds.has(n.id)).select(".label");
        highlightedLabels.style("display", "block");
        highlightedLabels.select(".name").text(n => n.name);
        highlightedLabels.select(".regcode").text(n => n.id);
        highlightedLabels.select(".profit").text(n => formatProfit(n.profit, '')).attr("fill", n => getProfitColor(n.profit));
    }

    function updateStats(groupNodes) {
        const count = groupNodes.length;
        const totalProfit = groupNodes.reduce((sum, node) => sum + (node.profit || 0), 0);
        statsCount.textContent = count;
        statsProfit.textContent = formatProfit(totalProfit, ' EUR');
        updateStatsTable(groupNodes);
        statsBox.classList.remove('hidden');
    }

    function updateStatsTable(nodes) {
        const getRank = (node) => {
            const isSource = sourceIds.has(node.id);
            const isTarget = targetIds.has(node.id);
            if (isSource && !isTarget) return 1;
            if (isSource && isTarget) return 2;
            return 3;
        };
        const sortedNodes = [...nodes].sort((a, b) => getRank(a) - getRank(b));
        tableBody.innerHTML = '';
        sortedNodes.forEach(node => {
            const row = document.createElement('tr');
            const profitColor = getProfitColor(node.profit);
            row.innerHTML = `
                <td>${node.name}</td>
                <td><a href="https://saraksts.lv/${node.id}" target="_blank" rel="noopener noreferrer">${node.id}</a></td>
                <td class="profit-cell" style="color: ${profitColor};">${formatProfit(node.profit, '')}</td>
            `;
            tableBody.appendChild(row);
        });
    }

    toggleButton.addEventListener('click', () => {
        tableContainer.classList.toggle('hidden');
        toggleButton.textContent = tableContainer.classList.contains('hidden') ? "Rādīt sarakstu" : "Slēpt sarakstu";
    });

    function formatProfit(profit, currency) {
        if (profit === null || isNaN(profit)) return "N/A";
        const sign = profit >= 0 ? '+' : '−';
        return `${sign} ${Math.abs(profit).toLocaleString('lv-LV')}${currency}`;
    }

    function getProfitColor(profit) {
        if (profit === null || isNaN(profit)) return "#555";
        return profit >= 0 ? "green" : "red";
    }

    function drag(simulation) {
        function dragstarted(event, d) { if (!event.active) simulation.alphaTarget(0.3).restart(); d.fx = d.x; d.fy = d.y; }
        function dragged(event, d) { d.fx = event.x; d.fy = event.y; }
        function dragended(event, d) { if (!event.active) simulation.alphaTarget(0); d.fx = null; d.fy = null; }
        return d3.drag().on("start", dragstarted).on("drag", dragged).on("end", dragended);
    }
});